#   Copyright 2019 Red Hat, Inc. All rights reserved.
#
#   Licensed under the Apache License, Version 2.0 (the "License"); you may
#   not use this file except in compliance with the License. You may obtain
#   a copy of the License at
#
#        http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#   WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#   License for the specific language governing permissions and limitations
#   under the License.
#
import copy
from unittest import mock

from osc_lib import exceptions

from octaviaclient.osc.v2 import constants
from octaviaclient.osc.v2 import l7policy
from octaviaclient.tests.unit.osc.v2 import constants as attr_consts
from octaviaclient.tests.unit.osc.v2 import fakes


class TestL7Policy(fakes.TestOctaviaClient):

    def setUp(self):
        super().setUp()

        self._l7po = fakes.createFakeResource('l7policy')
        self.l7po_info = copy.deepcopy(attr_consts.L7POLICY_ATTRS)
        self.columns = copy.deepcopy(constants.L7POLICY_COLUMNS)

        self.api_mock = mock.Mock()
        self.api_mock.l7policy_list.return_value = copy.deepcopy(
            {'l7policies': [attr_consts.L7POLICY_ATTRS]})

        lb_client = self.app.client_manager
        lb_client.load_balancer = self.api_mock


class TestL7PolicyList(TestL7Policy):

    def setUp(self):
        super().setUp()
        self.datalist = (tuple(
            attr_consts.L7POLICY_ATTRS[k] for k in self.columns),)
        self.cmd = l7policy.ListL7Policy(self.app, None)

    def test_l7policy_list_no_options(self):
        arglist = []
        verifylist = []

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        columns, data = self.cmd.take_action(parsed_args)

        self.api_mock.l7policy_list.assert_called_with()
        self.assertEqual(self.columns, columns)
        self.assertEqual(self.datalist, tuple(data))

    @mock.patch('octaviaclient.osc.v2.utils.get_l7policy_attrs')
    def test_l7policy_list_by_listener(self, mock_l7policy_attrs):
        mock_l7policy_attrs.return_value = {
            'listener_id': self._l7po.listener_id,
        }
        arglist = ['--listener', 'mock_li_id']
        verifylist = [('listener', 'mock_li_id')]

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        columns, data = self.cmd.take_action(parsed_args)

        self.api_mock.l7policy_list.assert_called_with(
            listener_id=self._l7po.listener_id
        )
        self.assertEqual(self.columns, columns)
        self.assertEqual(self.datalist, tuple(data))

    def test_l7policy_list_with_tags(self):
        arglist = ['--tags', 'foo,bar']
        verifylist = [('tags', ['foo', 'bar'])]
        expected_attrs = {
            'tags': ['foo', 'bar']
        }

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        columns, data = self.cmd.take_action(parsed_args)

        self.api_mock.l7policy_list.assert_called_with(**expected_attrs)
        self.assertEqual(self.columns, columns)
        self.assertEqual(self.datalist, tuple(data))

    def test_l7policy_list_with_any_tags(self):
        arglist = ['--any-tags', 'foo,bar']
        verifylist = [('any_tags', ['foo', 'bar'])]
        expected_attrs = {
            'tags-any': ['foo', 'bar']
        }

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        columns, data = self.cmd.take_action(parsed_args)

        self.api_mock.l7policy_list.assert_called_with(**expected_attrs)
        self.assertEqual(self.columns, columns)
        self.assertEqual(self.datalist, tuple(data))

    def test_l7policy_list_with_not_tags(self):
        arglist = ['--not-tags', 'foo,bar']
        verifylist = [('not_tags', ['foo', 'bar'])]
        expected_attrs = {
            'not-tags': ['foo', 'bar']
        }

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        columns, data = self.cmd.take_action(parsed_args)

        self.api_mock.l7policy_list.assert_called_with(**expected_attrs)
        self.assertEqual(self.columns, columns)
        self.assertEqual(self.datalist, tuple(data))

    def test_l7policy_list_with_not_any_tags(self):
        arglist = ['--not-any-tags', 'foo,bar']
        verifylist = [('not_any_tags', ['foo', 'bar'])]
        expected_attrs = {
            'not-tags-any': ['foo', 'bar']
        }

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        columns, data = self.cmd.take_action(parsed_args)

        self.api_mock.l7policy_list.assert_called_with(**expected_attrs)
        self.assertEqual(self.columns, columns)
        self.assertEqual(self.datalist, tuple(data))


class TestL7PolicyDelete(TestL7Policy):

    def setUp(self):
        super().setUp()
        self.cmd = l7policy.DeleteL7Policy(self.app, None)

    def test_l7policy_delete(self):
        arglist = [self._l7po.id]
        verifylist = [
            ('l7policy', self._l7po.id)
        ]

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        self.cmd.take_action(parsed_args)
        self.api_mock.l7policy_delete.assert_called_with(
            l7policy_id=self._l7po.id)

    @mock.patch('osc_lib.utils.wait_for_delete')
    def test_l7policy_delete_wait(self, mock_wait):
        arglist = [self._l7po.id, '--wait']
        verifylist = [
            ('l7policy', self._l7po.id),
            ('wait', True),
        ]

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        self.cmd.take_action(parsed_args)
        self.api_mock.l7policy_delete.assert_called_with(
            l7policy_id=self._l7po.id)
        mock_wait.assert_called_once_with(
            manager=mock.ANY,
            res_id=self._l7po.id,
            sleep_time=mock.ANY,
            status_field='provisioning_status')

    def test_l7policy_delete_failure(self):
        arglist = ['unknown_policy']
        verifylist = [
            ('l7policy', 'unknown_policy')
        ]
        self.api_mock.l7policy_list.return_value = {'l7policies': []}
        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        self.assertRaises(exceptions.CommandError, self.cmd.take_action,
                          parsed_args)
        self.assertNotCalled(self.api_mock.l7policy_delete)


class TestL7PolicyCreate(TestL7Policy):

    def setUp(self):
        super().setUp()
        self.api_mock.l7policy_create.return_value = {
            'l7policy': self.l7po_info}
        lb_client = self.app.client_manager
        lb_client.load_balancer = self.api_mock

        self.cmd = l7policy.CreateL7Policy(self.app, None)

    @mock.patch('octaviaclient.osc.v2.utils.get_l7policy_attrs')
    def test_l7policy_create(self, mock_attrs):
        mock_attrs.return_value = {
            'listener_id': self._l7po.listener_id,
            'name': self._l7po.name,
            'action': 'REDIRECT_TO_POOL',
            'redirect_pool_id': self._l7po.redirect_pool_id
        }
        arglist = ['mock_li_id',
                   '--name', self._l7po.name,
                   '--action', 'REDIRECT_TO_POOL'.lower(),
                   '--redirect-pool', self._l7po.redirect_pool_id]

        verifylist = [
            ('listener', 'mock_li_id'),
            ('name', self._l7po.name),
            ('action', 'REDIRECT_TO_POOL'),
            ('redirect_pool', self._l7po.redirect_pool_id)
        ]

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        self.cmd.take_action(parsed_args)
        self.api_mock.l7policy_create.assert_called_with(
            json={'l7policy': {
                'listener_id': self._l7po.listener_id,
                'name': self._l7po.name,
                'action': 'REDIRECT_TO_POOL',
                'redirect_pool_id': self._l7po.redirect_pool_id
            }})

    @mock.patch('osc_lib.utils.wait_for_status')
    @mock.patch('octaviaclient.osc.v2.utils.get_l7policy_attrs')
    def test_l7policy_create_wait(self, mock_attrs, mock_wait):
        mock_attrs.return_value = {
            'listener_id': self._l7po.listener_id,
            'name': self._l7po.name,
            'action': 'REDIRECT_TO_POOL',
            'redirect_pool_id': self._l7po.redirect_pool_id
        }
        self.api_mock.listener_show.return_value = {
            'loadbalancers': [{'id': 'mock_lb_id'}]}
        self.api_mock.l7policy_show.return_value = self.l7po_info
        arglist = ['mock_li_id',
                   '--name', self._l7po.name,
                   '--action', 'REDIRECT_TO_POOL'.lower(),
                   '--redirect-pool', self._l7po.redirect_pool_id,
                   '--wait']

        verifylist = [
            ('listener', 'mock_li_id'),
            ('name', self._l7po.name),
            ('action', 'REDIRECT_TO_POOL'),
            ('redirect_pool', self._l7po.redirect_pool_id),
            ('wait', True),
        ]

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        self.cmd.take_action(parsed_args)
        self.api_mock.l7policy_create.assert_called_with(
            json={'l7policy': {
                'listener_id': self._l7po.listener_id,
                'name': self._l7po.name,
                'action': 'REDIRECT_TO_POOL',
                'redirect_pool_id': self._l7po.redirect_pool_id
            }})
        mock_wait.assert_called_once_with(
            status_f=mock.ANY,
            res_id='mock_lb_id',
            sleep_time=mock.ANY,
            status_field='provisioning_status')

    @mock.patch('octaviaclient.osc.v2.utils.get_l7policy_attrs')
    def test_l7policy_create_with_tag(self, mock_attrs):
        mock_attrs.return_value = {
            'listener_id': self._l7po.listener_id,
            'name': self._l7po.name,
            'action': 'REDIRECT_TO_POOL',
            'redirect_pool_id': self._l7po.redirect_pool_id,
            'tags': ['foo']
        }

        arglist = ['mock_li_id',
                   '--name', self._l7po.name,
                   '--action', 'REDIRECT_TO_POOL'.lower(),
                   '--redirect-pool', self._l7po.redirect_pool_id,
                   '--tag', 'foo']

        verifylist = [
            ('listener', 'mock_li_id'),
            ('name', self._l7po.name),
            ('action', 'REDIRECT_TO_POOL'),
            ('redirect_pool', self._l7po.redirect_pool_id),
            ('tags', ['foo'])
        ]

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        self.cmd.take_action(parsed_args)
        self.api_mock.l7policy_create.assert_called_with(
            json={'l7policy': {
                'listener_id': self._l7po.listener_id,
                'name': self._l7po.name,
                'action': 'REDIRECT_TO_POOL',
                'redirect_pool_id': self._l7po.redirect_pool_id,
                'tags': ['foo']}})


class TestL7PolicyShow(TestL7Policy):

    def setUp(self):
        super().setUp()
        self.api_mock.l7policy_list.return_value = [{'id': self._l7po.id}]
        self.api_mock.l7policy_show.return_value = self.l7po_info
        lb_client = self.app.client_manager
        lb_client.load_balancer = self.api_mock

        self.cmd = l7policy.ShowL7Policy(self.app, None)

    @mock.patch('octaviaclient.osc.v2.utils.get_l7policy_attrs')
    def test_l7policy_show(self, mock_attrs):
        mock_attrs.return_value = {'l7policy_id': self._l7po.id}
        arglist = [self._l7po.id]
        verifylist = [
            ('l7policy', self._l7po.id),
        ]

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        self.cmd.take_action(parsed_args)
        self.api_mock.l7policy_show.assert_called_with(
            l7policy_id=self._l7po.id)


class TestL7PolicySet(TestL7Policy):

    def setUp(self):
        super().setUp()
        self.cmd = l7policy.SetL7Policy(self.app, None)

    def test_l7policy_set(self):
        arglist = [self._l7po.id, '--name', 'new_name']
        verifylist = [
            ('l7policy', self._l7po.id),
            ('name', 'new_name')
        ]

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        self.cmd.take_action(parsed_args)
        self.api_mock.l7policy_set.assert_called_with(
            self._l7po.id, json={'l7policy': {'name': 'new_name'}})

    @mock.patch('osc_lib.utils.wait_for_status')
    def test_l7policy_set_wait(self, mock_wait):
        arglist = [self._l7po.id, '--name', 'new_name', '--wait']
        verifylist = [
            ('l7policy', self._l7po.id),
            ('name', 'new_name'),
            ('wait', True),
        ]

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        self.cmd.take_action(parsed_args)
        self.api_mock.l7policy_set.assert_called_with(
            self._l7po.id, json={'l7policy': {'name': 'new_name'}})
        mock_wait.assert_called_once_with(
            status_f=mock.ANY,
            res_id=self._l7po.id,
            sleep_time=mock.ANY,
            status_field='provisioning_status')

    def test_l7policy_set_tag(self):
        self.api_mock.l7policy_show.return_value = {
            'tags': ['foo']
        }
        arglist = [self._l7po.id, '--tag', 'bar']
        verifylist = [
            ('l7policy', self._l7po.id),
            ('tags', ['bar']),
        ]

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        self.cmd.take_action(parsed_args)

        self.api_mock.l7policy_set.assert_called_once()
        kwargs = self.api_mock.l7policy_set.mock_calls[0][2]
        tags = kwargs['json']['l7policy']['tags']
        self.assertEqual(2, len(tags))
        self.assertIn('foo', tags)
        self.assertIn('bar', tags)

    def test_l7policy_set_tag_no_tag(self):
        self.api_mock.l7policy_show.return_value = {
            'tags': ['foo']
        }
        arglist = [self._l7po.id, '--tag', 'bar', '--no-tag']
        verifylist = [
            ('l7policy', self._l7po.id),
            ('tags', ['bar']),
        ]

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        self.cmd.take_action(parsed_args)

        self.api_mock.l7policy_set.assert_called_once_with(
            self._l7po.id,
            json={'l7policy': {'tags': ['bar']}}
        )


class TestL7PolicyUnset(TestL7Policy):
    PARAMETERS = ('name', 'description', 'redirect_http_code')

    def setUp(self):
        super().setUp()
        self.cmd = l7policy.UnsetL7Policy(self.app, None)

    def test_l7policy_unset_description(self):
        self._test_l7policy_unset_param('description')

    def test_l7policy_unset_name(self):
        self._test_l7policy_unset_param('name')

    def test_l7policy_unset_name_wait(self):
        self._test_l7policy_unset_param_wait('name')

    def test_l7policy_unset_redirect_http_code(self):
        self._test_l7policy_unset_param('redirect_http_code')

    def _test_l7policy_unset_param(self, param):
        self.api_mock.l7policy_set.reset_mock()
        arg_param = param.replace('_', '-') if '_' in param else param
        arglist = [self._l7po.id, '--%s' % arg_param]
        ref_body = {'l7policy': {param: None}}
        verifylist = [
            ('l7policy', self._l7po.id),
        ]
        for ref_param in self.PARAMETERS:
            verifylist.append((ref_param, param == ref_param))
        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        self.cmd.take_action(parsed_args)
        self.api_mock.l7policy_set.assert_called_once_with(
            self._l7po.id, json=ref_body)

    @mock.patch('osc_lib.utils.wait_for_status')
    def _test_l7policy_unset_param_wait(self, param, mock_wait):
        self.api_mock.l7policy_set.reset_mock()
        arg_param = param.replace('_', '-') if '_' in param else param
        arglist = [self._l7po.id, '--%s' % arg_param, '--wait']
        ref_body = {'l7policy': {param: None}}
        verifylist = [
            ('l7policy', self._l7po.id),
            ('wait', True),
        ]
        for ref_param in self.PARAMETERS:
            verifylist.append((ref_param, param == ref_param))
        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        self.cmd.take_action(parsed_args)
        self.api_mock.l7policy_set.assert_called_once_with(
            self._l7po.id, json=ref_body)
        mock_wait.assert_called_once_with(
            status_f=mock.ANY,
            res_id=self._l7po.id,
            sleep_time=mock.ANY,
            status_field='provisioning_status')

    def test_l7policy_unset_all(self):
        self.api_mock.l7policy_set.reset_mock()
        ref_body = {'l7policy': {x: None for x in self.PARAMETERS}}
        arglist = [self._l7po.id]
        for ref_param in self.PARAMETERS:
            arg_param = (ref_param.replace('_', '-') if '_' in ref_param else
                         ref_param)
            arglist.append('--%s' % arg_param)
        verifylist = list(zip(self.PARAMETERS, [True] * len(self.PARAMETERS)))
        verifylist = [('l7policy', self._l7po.id)] + verifylist
        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        self.cmd.take_action(parsed_args)
        self.api_mock.l7policy_set.assert_called_once_with(
            self._l7po.id, json=ref_body)

    def test_l7policy_unset_none(self):
        self.api_mock.l7policy_set.reset_mock()
        arglist = [self._l7po.id]
        verifylist = list(zip(self.PARAMETERS, [False] * len(self.PARAMETERS)))
        verifylist = [('l7policy', self._l7po.id)] + verifylist
        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        self.cmd.take_action(parsed_args)
        self.api_mock.l7policy_set.assert_not_called()

    def test_l7policy_unset_tag(self):
        self.api_mock.l7policy_set.reset_mock()
        self.api_mock.l7policy_show.return_value = {
            'tags': ['foo', 'bar']
        }

        arglist = [self._l7po.id, '--tag', 'foo']
        verifylist = [
            ('l7policy', self._l7po.id),
            ('tags', ['foo']),
        ]

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        self.cmd.take_action(parsed_args)

        self.api_mock.l7policy_set.assert_called_once_with(
            self._l7po.id,
            json={'l7policy': {'tags': ['bar']}}
        )

    def test_l7policy_unset_all_tag(self):
        self.api_mock.l7policy_set.reset_mock()
        self.api_mock.l7policy_show.return_value = {
            'tags': ['foo', 'bar']
        }

        arglist = [self._l7po.id, '--all-tag']
        verifylist = [
            ('l7policy', self._l7po.id),
            ('all_tag', True),
        ]

        parsed_args = self.check_parser(self.cmd, arglist, verifylist)
        self.cmd.take_action(parsed_args)

        self.api_mock.l7policy_set.assert_called_once_with(
            self._l7po.id,
            json={'l7policy': {'tags': []}}
        )
